<?php
/**
 * Plugin Name: Broadercaster Funeral Services
 * Description: Complete funeral services solution - displays service lists and individual service details via iframes with auto-height adjustment
 * Version: 2.1
 * Date: 25/11/2025
 * Author: Dan Liu
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class BroaderCasterFuneralPlugin {

    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_save_data', array($this, 'save_data'));
        add_action('wp_ajax_get_data', array($this, 'get_data'));
        add_action('wp_ajax_nopriv_get_data', array($this, 'get_data'));

        // Register both shortcodes
        add_shortcode('broadercaster_funeral_service_tab', array($this, 'render_services_list_shortcode'));
        add_shortcode('broadercaster_service_view', array($this, 'render_service_view_shortcode'));
    }

    public function init() {
        // Plugin initialization

        // Set default URL if not configured
        if (get_option('broadercaster_funeralservice_url') === false) {
            update_option('broadercaster_funeralservice_url', 'https://funeralservices.broadercaster.app/api');
        }
    }

    // Add admin menu
    public function add_admin_menu() {
        add_options_page(
            'Broadercaster Funerals Settings',
            'Broadercaster Funerals',
            'manage_options',
            'broadercaster-funeral-plugin',
            array($this, 'admin_page')
        );
    }

    // Enqueue scripts for frontend
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        wp_enqueue_script(
            'broadercaster-iframe-handler',
            plugin_dir_url(__FILE__) . 'assets/iframe-handler.js',
            array('jquery'),
            '2.0',
            true
        );

        // Localize script for AJAX (legacy support)
        wp_localize_script('broadercaster-iframe-handler', 'broadercasterFuneralService', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('funeral_services_nonce')
        ));
    }

    // Enqueue scripts for admin
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'settings_page_broadercaster-funeral-plugin') {
            return;
        }

        wp_enqueue_script('jquery');
        wp_enqueue_script(
            'broadercaster-admin-js',
            plugin_dir_url(__FILE__) . 'assets/admin.js',
            array('jquery'),
            '2.0',
            true
        );

        wp_localize_script('broadercaster-admin-js', 'broadercasterFuneralServiceAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('broadercaster_funeralservice_admin_nonce')
        ));
    }

    // Admin page HTML
    public function admin_page() {
        $token = get_option('broadercaster_funeralservice_token', '');
        $division = get_option('broadercaster_funeralservice_division', '');
        $section = get_option('broadercaster_funeralservice_section', '');
        $url = get_option('broadercaster_funeralservice_url', '');
        
        ?>
        <div class="wrap">
            <h1>BroaderCaster Settings</h1>
            <form id="api-tabs-settings">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="url">Server URL</label>
                        </th>
                        <td>
                            <input type="text"
                                   id="url"
                                   name="url"
                                   value="<?php echo esc_attr($url); ?>"
                                   class="regular-text"
                                   placeholder="Enter server url" />
                            <p class="description">
                                This URL is used to retrieve your service information.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="token">Token</label>
                        </th>
                        <td>
                            <input type="text"
                                   id="token"
                                   name="token"
                                   value="<?php echo esc_attr($token); ?>"
                                   class="regular-text"
                                   placeholder="Enter your token" />
                            <p class="description">
                                This token will be used to authenticate API requests.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="division">Division</label>
                        </th>
                        <td>
                            <input type="text"
                                   id="division"
                                   name="division"
                                   value="<?php echo esc_attr($division); ?>"
                                   class="regular-text"
                                   placeholder="Enter your division" />
                            <p class="description">
                                This will be used to identify service division.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="section">Section</label>
                        </th>
                        <td>
                            <input type="text"
                                   id="section"
                                   name="section"
                                   value="<?php echo esc_attr($section); ?>"
                                   class="regular-text"
                                   placeholder="Enter your section" />
                            <p class="description">
                                This will be used to identify service section.
                            </p>
                        </td>
                    </tr>
                </table>
                <p class="submit">
                    <button type="submit" class="button button-primary">Save Settings</button>
                </p>
            </form>

            <div id="save-message" style="display:none;"></div>

            <hr>
            <h2>Usage</h2>
            <h3>Services List</h3>
            <p>To display the funeral services list (live, upcoming, and completed services), use this shortcode:</p>
            <code>[broadercaster_funeral_service_tab]</code>
            <p>Customization options:</p>
            <code>[broadercaster_funeral_service_tab min_height="600px" border="1px solid #ccc" border_radius="12px"]</code>

            <hr>
            <h3>Individual Service View</h3>
            <p>To display an individual service detail page, use this shortcode:</p>
            <code>[broadercaster_service_view]</code>
            <p><strong>Important:</strong> The page must be accessed with a <code>uuid</code> URL parameter (e.g., <code>?uuid=abc123</code>).</p>
            <p>Customization options:</p>
            <code>[broadercaster_service_view min_height="800px" border="1px solid #ddd" border_radius="8px"]</code>
        </div>
        <?php
    }

    // Save settings via AJAX
    public function save_data() {
        // Check for both possible nonce fields
        $nonce_verified = false;
        if (isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'broadercaster_funeralservice_admin_nonce')) {
            $nonce_verified = true;
        } elseif (isset($_POST['admin_nonce']) && wp_verify_nonce($_POST['admin_nonce'], 'broadercaster_funeralservice_admin_nonce')) {
            $nonce_verified = true;
        }

        if (!$nonce_verified || !current_user_can('manage_options')) {
            wp_send_json_error('Security check failed');
            return;
        }

        $url = isset($_POST['url']) ? sanitize_text_field($_POST['url']) : '';
        $token = isset($_POST['token']) ? sanitize_text_field($_POST['token']) : '';
        $division = isset($_POST['division']) ? sanitize_text_field($_POST['division']) : '';
        $section = isset($_POST['section']) ? sanitize_text_field($_POST['section']) : '';

        // Update options
        update_option('broadercaster_funeralservice_url', $url);
        update_option('broadercaster_funeralservice_token', $token);
        update_option('broadercaster_funeralservice_division', $division);
        update_option('broadercaster_funeralservice_section', $section);

        wp_send_json_success('Settings saved successfully!');
    }

    // Handle API data requests (legacy support for direct API calls)
    public function get_data() {
        if (!wp_verify_nonce($_POST['nonce'], 'funeral_services_nonce')) {
            wp_send_json_error('Security check failed');
        }

        $tab = sanitize_text_field($_POST['tab']);
        $token = get_option('broadercaster_funeralservice_token', '');
        $division = get_option('broadercaster_funeralservice_division', '');
        $section = get_option('broadercaster_funeralservice_section', '');
        $url = get_option('broadercaster_funeralservice_url', '');
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $rowPerPage = 20;

        if (empty($url)) {
            wp_send_json_error('Broadercaster server URL not configured');
        }

        if (empty($token)) {
            wp_send_json_error('Broadercaster API token not configured');
        }

        $paginationParameter = '?page=' . $page . '&rowPerPage=' . $rowPerPage;

        // Define API endpoints for each tab
        $endpoints = array(
            'tab-inprogress' => $url . '/OrganizationFuneralEvent/live' . $paginationParameter,
            'tab-upcoming' => $url . '/OrganizationFuneralEvent/upcoming' . $paginationParameter,
            'tab-completed' => $url . '/OrganizationFuneralEvent/completed' . $paginationParameter
        );

        if (!isset($endpoints[$tab])) {
            wp_send_json_error('Invalid tab');
        }

        // Make API request
        $response = wp_remote_get($endpoints[$tab], array(
            'headers' => array(
                'sls-auth-token' => $token
            ),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            wp_send_json_error('API request failed: ' . $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error('Invalid JSON response');
        }

        wp_send_json_success($data);
    }

    /**
     * Shortcode: [broadercaster_funeral_service_tab]
     * Displays the services list iframe (live, upcoming, completed services)
     */
    public function render_services_list_shortcode($atts) {
        $token = get_option('broadercaster_funeralservice_token', '');
        $division = get_option('broadercaster_funeralservice_division', '');
        $url = get_option('broadercaster_funeralservice_url', '');
        $section = get_option('broadercaster_funeralservice_section', '');

        if (empty($url)) {
            return '<div class="broadercaster-error">Broadercaster server URL not configured. Please configure it in the admin panel.</div>';
        }

        if (empty($token)) {
            return '<div class="broadercaster-error">API token not configured. Please configure it in the admin panel.</div>';
        }

        // Parse shortcode attributes for customization
        $atts = shortcode_atts(array(
            'min_height' => '400px',
            'border' => 'none',
            'border_radius' => '8px'
        ), $atts);

        $iframe_url = esc_url($url . '/eventsfrontend/services.php?token=' . $token . (!empty($division) ? '&divisionuuid=' . urlencode($division) : '') . (!empty($section) ? '&sectionuuid=' . urlencode($section) : ''));
        $unique_id = 'services-' . uniqid();

        return $this->render_iframe_container($iframe_url, $unique_id, $atts, 'Broadercaster Services');
    }

    /**
     * Shortcode: [broadercaster_service_view]
     * Displays an individual service detail iframe
     */
    public function render_service_view_shortcode($atts) {
        // Parse shortcode attributes for customization
        $atts = shortcode_atts(array(
            'min_height' => '400px',
            'border' => 'none',
            'border_radius' => '8px',
            'base_url' => 'https://broadercaster.app/eventsfrontend/service_view.php'
        ), $atts);

        // Get UUID from URL parameter
        $uuid = isset($_GET['uuid']) ? sanitize_text_field($_GET['uuid']) : '';

        // If no UUID provided, show error message
        if (empty($uuid)) {
            return '<div class="broadercaster-error" style="padding: 20px; background-color: #fff3cd; border: 1px solid #ffc107; border-radius: 4px; color: #856404;">
                <p><strong>Notice:</strong> No service UUID provided. Please access this page with a valid UUID parameter (e.g., ?uuid=your-service-id).</p>
            </div>';
        }

        // Build the iframe URL
        $iframe_url = esc_url(add_query_arg('uuid', $uuid, $atts['base_url']));
        $unique_id = 'service-view-' . sanitize_key($uuid);

        return $this->render_iframe_container($iframe_url, $unique_id, $atts, 'Broadercaster Service View');
    }

    /**
     * Shared method to render iframe container with auto-height adjustment
     */
    private function render_iframe_container($iframe_url, $unique_id, $atts, $title) {
        ob_start();
        ?>
        <style>
            .broadercaster-iframe-container {
                width: 100%;
                max-width: 100%;
                min-height: <?php echo esc_attr($atts['min_height']); ?>;
                margin: 20px 0;
                border: <?php echo esc_attr($atts['border']); ?>;
                border-radius: <?php echo esc_attr($atts['border_radius']); ?>;
                overflow: hidden;
                box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
                box-sizing: border-box;
                margin-left: 0;
                margin-right: 0;
                padding: 0;
                position: relative;
            }

            .broadercaster-iframe {
                width: 100%;
                height: 100%;
                border: none;
                display: block;
                max-width: 100%;
                box-sizing: border-box;
                overflow: hidden;
            }

            .broadercaster-loading {
                display: flex;
                justify-content: center;
                align-items: center;
                min-height: <?php echo esc_attr($atts['min_height']); ?>;
                width: 100%;
                background-color: #f9f9f9;
                color: #666;
                font-family: Arial, sans-serif;
                box-sizing: border-box;
            }

            .broadercaster-loading-spinner {
                border: 4px solid #f3f3f3;
                border-top: 4px solid #3498db;
                border-radius: 50%;
                width: 40px;
                height: 40px;
                animation: broadercaster-spin 1s linear infinite;
                margin-right: 10px;
                flex-shrink: 0;
            }

            @keyframes broadercaster-spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }

            /* Responsive adjustments */
            @media (max-width: 768px) {
                .broadercaster-iframe-container {
                    margin: 15px 0;
                }
            }

            @media (max-width: 480px) {
                .broadercaster-iframe-container {
                    margin: 10px 0;
                }
            }

            .broadercaster-iframe-container * {
                box-sizing: border-box;
            }

            /* Override any potential theme max-width restrictions */
            .entry-content .broadercaster-iframe-container,
            .post-content .broadercaster-iframe-container,
            .page-content .broadercaster-iframe-container,
            .content .broadercaster-iframe-container {
                max-width: none !important;
                width: 100% !important;
            }
        </style>

        <!-- Broadercaster Iframe Container -->
        <div class="broadercaster-iframe-container" id="broadercaster-container-<?php echo esc_attr($unique_id); ?>">
            <div class="broadercaster-loading" id="broadercaster-loading-<?php echo esc_attr($unique_id); ?>">
                <div class="broadercaster-loading-spinner"></div>
                <span>Loading...</span>
            </div>
            <iframe
                id="broadercaster-iframe-<?php echo esc_attr($unique_id); ?>"
                class="broadercaster-iframe"
                src="<?php echo $iframe_url; ?>"
                style="display: none;"
                scrolling="no"
                title="<?php echo esc_attr($title); ?>">
            </iframe>
        </div>

        <script>
            (function() {
                var uniqueId = '<?php echo esc_js($unique_id); ?>';
                var iframe = document.getElementById('broadercaster-iframe-' + uniqueId);
                var container = document.getElementById('broadercaster-container-' + uniqueId);
                var loading = document.getElementById('broadercaster-loading-' + uniqueId);

                // Listen for messages from the iframe
                window.addEventListener('message', function(event) {
                    // Verify the message is from our iframe
                    if (event.source !== iframe.contentWindow) {
                        return;
                    }

                    // Handle height update messages
                    if (event.data && event.data.type === 'iframeHeight') {
                        var height = event.data.height;
                        if (height && height > 0) {
                            iframe.style.height = height + 'px';
                            container.style.height = height + 'px';
                        }
                    }
                }, false);

                // Show iframe when loaded
                iframe.addEventListener('load', function() {
                    loading.style.display = 'none';
                    iframe.style.display = 'block';

                    // Request initial height from iframe
                    iframe.contentWindow.postMessage({type: 'requestHeight'}, '*');
                });

                // Handle iframe loading errors
                iframe.addEventListener('error', function() {
                    loading.innerHTML =
                        '<div style="text-align: center; color: #e74c3c;">' +
                        '<p>Failed to load content. Please try refreshing the page.</p>' +
                        '</div>';
                });

                // Periodically check for height changes (fallback)
                setInterval(function() {
                    if (iframe.style.display !== 'none') {
                        iframe.contentWindow.postMessage({type: 'requestHeight'}, '*');
                    }
                }, 500);
            })();
        </script>
        <?php
        return ob_get_clean();
    }
}

// Initialize the plugin
new BroaderCasterFuneralPlugin();

// Create plugin directory structure on activation
register_activation_hook(__FILE__, 'broadercaster_create_plugin_assets');

function broadercaster_create_plugin_assets() {
    $plugin_dir = plugin_dir_path(__FILE__);
    $assets_dir = $plugin_dir . 'assets/';

    if (!file_exists($assets_dir)) {
        wp_mkdir_p($assets_dir);
    }
}
?>
